<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 09.01.14 16:27
 */

namespace VM\FinancialStatementsBundle\Services\Reports;


use VM\FinancialStatementsBundle\Services\Reports\Svg\SvgChar;
use VM\FinancialStatementsBundle\Services\Reports\Svg\SvgChart;

class SimpleReport
{
    protected $topMargin = 20;


    protected $leftMargin = 30;


    /**
     * @var GamePdf
     */
    protected $Pdf;

    /**
     * @var \stdClass
     */
    protected $Data;


    public function __construct(\stdClass $Data)
    {
        $this->Data = $Data;
        $this->init();
        $this->setDefaultFont();

        $this->Pdf->AddPage('L', 'A4', true);
        $this->drawTitle();
        $this->drawTable();

        $this->Pdf->AddPage('L', 'A4', true);

        $this->drawChart("Przychody netto ze sprzedaży towarów i usług (w zł)", $this->Data->rachunekZyskowIStrat->przychodyNettoZeSprzedazy);
        $this->drawChart("Zysk netto (w zł)", $this->Data->rachunekZyskowIStrat->zyskNetto);

    }

    public function getPdf()
    {
        return $this->Pdf;
    }

    protected function init()
    {
        $this->Pdf = new GamePdf('L', 'px');
        $this->Pdf->SetMargins($this->leftMargin, $this->topMargin, $this->leftMargin, true);
        $this->Pdf->setHeaderMargin(120);
        $this->Pdf->setHeaderData('',0, $this->Data->companyName, 'Raport uproszczony');
    }



    protected function drawTitle()
    {
        $this->Pdf->MultiCell($this->Pdf->getPageWidth() - 2 * $this->leftMargin, 5, 'Prognoza przepływów finansowych', 0, 'L', 0, 1, '', '', true);
    }


    protected function drawTable()
    {
        $firstColumnWidth = 60;
        $secondColumnWidth = 200;
        $maxNumberOfValueColumns = 7;
        $columnValueWidth = ($this->Pdf->getPageWidth() - 2 * $this->leftMargin - $firstColumnWidth - $secondColumnWidth) / $maxNumberOfValueColumns;

        $cellHeight = 20;
        $firstYear = $this->getFirstYear();

        $this->setSmallerFont();

        // headers
        $this->Pdf->Ln($cellHeight);
        $this->setThBgColor();
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->Pdf->MultiCell($secondColumnWidth, 2 * $cellHeight, 'Kategoria finansowa', 1, 'C', true, 0, '', '', true, 0, false, true, 2 * $cellHeight, 'M', false);
        $this->Pdf->MultiCell($columnValueWidth * $maxNumberOfValueColumns, $cellHeight, 'Zmiana stanu środków pieniężnych (PLN)', 1, 'C', true, 1, '', '', true, 0, false, true, $cellHeight, 'M', false);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth + $secondColumnWidth);
        $round = 1;
        for($i = $firstYear; $i < $firstYear + $maxNumberOfValueColumns; $i++)
        {
            $last = $i == $firstYear + $maxNumberOfValueColumns - 1;
            $this->Pdf->MultiCell($columnValueWidth, $cellHeight, 'Tura ' . $round, 1, 'C', true, $last, '', '', true, 0, false, true, $cellHeight, 'M', false);
            $round++;
        }

        // first row
        $this->setTdBgColor();
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Stan środków pieniężnych na początek roku', $this->Data->stanSrodkowNaPoczatekRoku, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);

        //first section
        $this->Pdf->SetX($this->leftMargin);
        $this->Pdf->MultiCell($firstColumnWidth, 9 * $cellHeight, 'rachunek zysków i strat', 1, 'C', true, 0, '', '', true, 0, false, true, 9 * $cellHeight, 'M', false);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Przychody netto i zrównane z nimi', $this->Data->rachunekZyskowIStrat->przychodyNettoZeSprzedazy, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Koszty zmienne', $this->Data->rachunekZyskowIStrat->kosztyZmienne, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Koszty stałe', $this->Data->rachunekZyskowIStrat->kosztyStale, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('w tym amortyzacja', $this->Data->rachunekZyskowIStrat->amortyzacja, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Przychody finansowe', $this->Data->rachunekZyskowIStrat->przychodyFinansowe, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Koszty finansowe', $this->Data->rachunekZyskowIStrat->kosztyFinansowe, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Zysk brutto', $this->Data->rachunekZyskowIStrat->zyskBrutto, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Podatek dochodowy', $this->Data->rachunekZyskowIStrat->podatekDochodowy, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Zysk netto', $this->Data->rachunekZyskowIStrat->zyskNetto, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);

        //second section
        $this->Pdf->SetX($this->leftMargin);
        $this->Pdf->MultiCell($firstColumnWidth, 4 * $cellHeight, 'rachunek przepływów pieniężnych', 1, 'C', true, 0, '', '', true, 0, false, true, 4 * $cellHeight, 'M', false);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Nakłady na kapitał pracujący', $this->Data->rachunekPrzeplywowPienieznych->nakladyNaKapitalPracujacy, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Nakłady inwestycyjne', $this->Data->rachunekPrzeplywowPienieznych->nakladyInwestycyjne, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Wpływy finansowe', $this->Data->rachunekPrzeplywowPienieznych->zaciagniecieKredytowIPozyczek, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Wydatki finansowe', $this->Data->rachunekPrzeplywowPienieznych->splataKredytowIPozyczek, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Stan wyrobów gotowych na magazynie', $this->Data->stanWyrobowGotowych, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
        $this->Pdf->SetX($this->leftMargin + $firstColumnWidth);
        $this->drawRow('Stan środków pieniężnych na koniec okresu', $this->Data->stanSrodkowNaKoniecRoku, $cellHeight, $secondColumnWidth, $columnValueWidth, $maxNumberOfValueColumns);
    }

    protected function drawRow($label, $data, $cellHeight, $labelWidth, $columnValueWidth, $maxNumberOfValueColumns)
    {
        $firstYear = $this->getFirstYear();
        $this->Pdf->MultiCell($labelWidth, $cellHeight, $label, 1, 'L', true, 0, '', '', true, 0, false, true, $cellHeight, 'M', true);

        for($i = $firstYear; $i < $firstYear + $maxNumberOfValueColumns; $i++)
        {
            $last = ($i == $firstYear + $maxNumberOfValueColumns - 1) ? 1: 0;
            $value = (isset($data[$i])) ? $this->roundPrice($data[$i]) : '';
            $this->Pdf->MultiCell($columnValueWidth, $cellHeight, $value, 1, 'R', true, $last, '', '', true, 0, false, true, $cellHeight, 'M', true);
        }

    }


    protected function drawChart($title, $values)
    {
        $data = array();
        foreach($values as $i => $val)
        {
            $data[] = array(
                'label' => 'Tura ' . $i,
                'value' => $val
            );
        }

        $this->Pdf->MultiCell($this->Pdf->getPageWidth() - 2 * $this->leftMargin, 5, $title, 0, 'C', 0, 1, '', '', true);
        $Chart = new SvgChart();
        $image = $Chart->create($data);

        $this->Pdf->ImageSVGFromString($image, $this->leftMargin ,$this->Pdf->GetY(), 800, 200);
        $this->Pdf->Ln(210);
    }


    protected function getNumberOfYears()
    {
        return count($this->Data->stanSrodkowNaPoczatekRoku);
    }


    protected function getFirstYear()
    {
        $years = array_keys($this->Data->stanSrodkowNaPoczatekRoku);
        return array_shift($years);
    }



    protected function setDefaultFont()
    {
        $this->Pdf->SetFontSize(10);
        $this->Pdf->SetFont($this->Pdf->_defaultFontFamily);
    }

    protected function setSmallerFont()
    {
        $this->Pdf->SetFontSize(8);
        $this->Pdf->SetFont($this->Pdf->_defaultFontFamily);
    }


    protected function setThBgColor()
    {
        $this->Pdf->SetFillColorArray(array(119, 195, 253));
    }

    protected function setTdBgColor()
    {
        $this->Pdf->SetFillColorArray(array(255, 255, 255));
    }

    protected function roundPrice($price)
    {
        return number_format($price, 2, ',', ' ');
    }
}
    